/*------------------------------------------------------------------------------
Name:      XmlBlasterAccess.cpp
Project:   xmlBlaster.org
Copyright: xmlBlaster.org, see xmlBlaster-LICENSE file
------------------------------------------------------------------------------*/

#include <client/XmlBlasterAccess.h>
#include <util/Global.h>
#include <util/lexical_cast.h>
#include <util/Timestamp.h>
#include <util/dispatch/DispatchManager.h>

namespace org { namespace xmlBlaster { namespace client {

using namespace std;
using namespace org::xmlBlaster::util;
using namespace org::xmlBlaster::util::qos;
using namespace org::xmlBlaster::util::dispatch;
using namespace org::xmlBlaster::util::dispatch;
using namespace org::xmlBlaster::util::qos::storage;
using namespace org::xmlBlaster::util::qos::address;
using namespace org::xmlBlaster::authentication;
using namespace org::xmlBlaster::client::protocol;
using namespace org::xmlBlaster::client::key;
using namespace org::xmlBlaster::client::qos;

XmlBlasterAccess::XmlBlasterAccess(Global& global)
   : ME(string("XmlBlasterAccess-UNCONNECTED")),
     serverNodeId_("xmlBlaster"), 
     connectQos_(global), 
     connectReturnQos_(global),
     global_(global), 
     log_(global.getLog("org.xmBlaster.client"))
{
   log_.call(ME, "::constructor");
   cbServer_           = NULL;
   updateClient_       = NULL;
   connection_         = NULL;
   dispatchManager_    = NULL;
   connectionProblems_ = NULL;
   instanceName_       = lexical_cast<std::string>(TimestampFactory::getInstance().getTimestamp());
}

XmlBlasterAccess::~XmlBlasterAccess()
{
   if (log_.call()) log_.call(ME, "destructor");
   if (cbServer_) {
      CbQueueProperty prop = connectQos_.getSessionCbQueueProperty(); // Creates a default property for us if none is available
      CallbackAddress addr = prop.getCurrentCallbackAddress(); // c++ may not return null
      global_.getCbServerPluginManager().releasePlugin( instanceName_, addr.getType(), addr.getVersion() );
      cbServer_ = NULL;
   }
   if (log_.trace()) log_.trace(ME, "destructor: going to delete the connection");
   if (connection_) {
      connection_->shutdown();
      delete connection_;
      connection_ = NULL;
   }
   dispatchManager_    = NULL;
   updateClient_       = NULL;
   connectionProblems_ = NULL;
   if (log_.trace()) log_.trace(ME, "destructor ended");
}


ConnectReturnQos XmlBlasterAccess::connect(const ConnectQos& qos, I_Callback *clientAddr)
{
   ME = string("XmlBlasterAccess-") + qos.getSessionQos().getAbsoluteName();
   if (log_.call()) log_.call(ME, "::connect");
   if (log_.dump()) log_.dump(ME, string("::connect: qos: ") + qos.toXml());

   //global_.setId(loginName + currentTimeMillis()); // Not secure if two clients start simultaneously

   connectQos_ = qos;
   SecurityQos securityQos = connectQos_.getSecurityQos();
//   initSecuritySettings(securityQos.getPluginType(), securityQos.getPluginVersion());
   ME = string("XmlBlasterAccess-") + getId();
   string typeVersion = global_.getProperty().getStringProperty("queue/defaultPlugin", "CACHE,1.0");
   typeVersion = global_.getProperty().getStringProperty("queue/connection/defaultPlugin", "typeVersion");
   //string queueId = string("connection:") + getId();
   updateClient_ = clientAddr;
   //if (!cbServer_) createDefaultCbServer();
   createDefaultCbServer();

   if (log_.trace()) log_.trace(ME, string("::connect. CbServer done"));
   // currently the simple version will do it ...
   if (!dispatchManager_) dispatchManager_ = &(global_.getDispatchManager());

/*
   string type = connectQos_.getAddress().getType();
   string version = "1.0";
   connection_ = &(dispatchManager_->getPlugin(type, version));
*/
   if (!connection_) {
      connection_ = dispatchManager_->getConnectionsHandler(instanceName_);
   }

   if (connectionProblems_) {
      connection_->initFailsafe(connectionProblems_);
      connectionProblems_ = NULL;
   }
   if (log_.trace()) log_.trace(ME, string("::connect. connectQos: ") + connectQos_.toXml());
   connectReturnQos_ = connection_->connect(connectQos_);

   ME = string("XmlBlasterAccess-") + connectReturnQos_.getSessionQos().getAbsoluteName();
   
   // Is done in ConnectionsHandler.cpp
   //global_.setId(connectReturnQos_.getSessionQos().getAbsoluteName());
   
   return connectReturnQos_;
}

void XmlBlasterAccess::createDefaultCbServer()
{
   log_.call(ME, "::createDefaultCbServer");

   CbQueueProperty prop = connectQos_.getSessionCbQueueProperty(); // Creates a default property for us if none is available
   CallbackAddress addr = prop.getCurrentCallbackAddress(); // c++ may not return null

   if(!cbServer_)
     cbServer_ = initCbServer(getLoginName(), addr.getType(), addr.getVersion());

   addr.setAddress(cbServer_->getCbAddress());
   addr.setType(cbServer_->getCbProtocol());
   prop.setCallbackAddress(addr);
   connectQos_.setSessionCbQueueProperty(prop);
   if (log_.trace()) log_.trace(ME, string("::createDefaultCbServer: connectQos: ") + connectQos_.toXml());
   log_.info(ME, "Callback settings: " + prop.getSettings());
}

I_CallbackServer*
XmlBlasterAccess::initCbServer(const string& loginName, const string& type, const string& version)
{
   if (log_.call()) log_.call(ME, string("::initCbServer: loginName='") + loginName + "' type='" + type + "' version='" + version +"'");
   if (log_.trace()) log_.trace(ME, string("Using 'client.cbProtocol=") + type + string("' to be used by ") + getServerNodeId() + string(", trying to create the callback server ..."));
   I_CallbackServer* server = &(global_.getCbServerPluginManager().getPlugin(instanceName_, type, version));
   if (log_.trace()) log_.trace(ME, "After callback plugin creation");
   server->initialize(loginName, *this);
   if (log_.trace()) log_.trace(ME, "After callback plugin initialize");
   return server;
}

void
XmlBlasterAccess::initSecuritySettings(const string& /*secMechanism*/, const string& /*secVersion*/)
{
   log_.error(ME, "initSecuritySettings not implemented yet");
}

bool
XmlBlasterAccess::disconnect(const DisconnectQos& qos, bool flush, bool shutdown, bool shutdownCb)
{
   bool ret1 = true;
   bool ret3 = true;
   if (log_.call()) {
      log_.call(ME, string("disconnect called with flush='") + Global::getBoolAsString(flush) + 
                              "' shutdown='" + Global::getBoolAsString(shutdown) + 
                    "' shutdownCb='" + Global::getBoolAsString(shutdownCb) + "'");
   }

   if (log_.trace()) log_.trace(ME, "disconnecting the client connection");
   if (log_.dump()) log_.dump(ME, string("disconnect: the qos is:\n") + qos.toXml());
   if (connection_ != NULL) {
      ret1  = connection_->disconnect(qos);
      if (shutdown) connection_->shutdown();
   }
   else {
      ret1 = false;
   }
   if (shutdownCb) {
      if (cbServer_) {
         ret3 = cbServer_->shutdownCb();

         CbQueueProperty prop = connectQos_.getSessionCbQueueProperty(); // Creates a default property for us if none is available
         CallbackAddress addr = prop.getCurrentCallbackAddress(); // c++ may not return null
         global_.getCbServerPluginManager().releasePlugin( instanceName_, addr.getType(), addr.getVersion() );
         cbServer_ = NULL;
      }
      else ret3 = false;
   }
   return ret1 && ret3;
}

string XmlBlasterAccess::getId()
{
   return getSessionName();
}

string XmlBlasterAccess::getSessionName()
{
   string ret = connectReturnQos_.getSessionQos().getAbsoluteName();
   if (ret == "") ret = connectQos_.getSessionQos().getAbsoluteName();
   return ret;
}

string XmlBlasterAccess::getLoginName()
{
   try {
      string nm = connectQos_.getSecurityQos().getUserId();
      if (nm != "") return nm;
   }
   catch (XmlBlasterException e) {
      log_.warn(ME, e.toString());
   }
   return string("client?");
}

void XmlBlasterAccess::setServerNodeId(const string& nodeId)
{
   serverNodeId_ = nodeId;
}

string XmlBlasterAccess::getServerNodeId() const
{
   return serverNodeId_;
}

/*
MsgQueueEntry
XmlBlasterAccess::queueMessage(const MsgQueueEntry& entry)
{
 return entry;
}

vector<MsgQueueEntry*>
XmlBlasterAccess::queueMessage(const vector<MsgQueueEntry*>& entries)
{
   return entries;
}
*/

SubscribeReturnQos XmlBlasterAccess::subscribe(const SubscribeKey& key, const SubscribeQos& qos)
{
   if (log_.call()) log_.call(ME, "subscribe");
   if (log_.dump()) {
      log_.dump(ME, string("subscribe. The key:\n") + key.toXml());
      log_.dump(ME, string("subscribe. The Qos:\n") + qos.toXml());
   }
   return connection_->subscribe(key, qos);
}

vector<MessageUnit> XmlBlasterAccess::get(const GetKey& key, const GetQos& qos)
{
   if (log_.call()) log_.call(ME, "get");
   if (log_.dump()) {
      log_.dump(ME, string("get. The key:\n") + key.toXml());
      log_.dump(ME, string("get. The Qos:\n") + qos.toXml());
   }
   return connection_->get(key, qos);
}

vector<UnSubscribeReturnQos> 
XmlBlasterAccess::unSubscribe(const UnSubscribeKey& key, const UnSubscribeQos& qos)
{
   if (log_.call()) log_.call(ME, "unSubscribe");
   if (log_.dump()) {
      log_.dump(ME, string("unSubscribe. The key:\n") + key.toXml());
      log_.dump(ME, string("unSubscribe. The Qos:\n") + qos.toXml());
   }
   return connection_->unSubscribe(key, qos);
}

PublishReturnQos XmlBlasterAccess::publish(const MessageUnit& msgUnit)
{
   if (log_.call()) log_.call(ME, "publish");
   if (log_.dump()) {
      log_.dump(ME, string("publish. The msgUnit:\n") + msgUnit.toXml());
   }
   return connection_->publish(msgUnit);
}

void XmlBlasterAccess::publishOneway(const vector<MessageUnit>& msgUnitArr)
{
   if (log_.call()) log_.call(ME, "publishOneway");
   if (log_.dump()) {
      for (vector<MessageUnit>::size_type i=0; i < msgUnitArr.size(); i++) {
         log_.dump(ME, string("publishOneway. The msgUnit[") + lexical_cast<std::string>(i) + "]:\n" + msgUnitArr[i].toXml());
      }
   }
   connection_->publishOneway(msgUnitArr);
}

vector<PublishReturnQos> XmlBlasterAccess::publishArr(const vector<MessageUnit> &msgUnitArr)
{
   if (log_.call()) log_.call(ME, "publishArr");
   if (log_.dump()) {
      for (vector<MessageUnit>::size_type i=0; i < msgUnitArr.size(); i++) {
         log_.dump(ME, string("publishArr. The msgUnit[") + lexical_cast<std::string>(i) + "]:\n" + msgUnitArr[i].toXml());
      }
   }
   return connection_->publishArr(msgUnitArr);
}

vector<EraseReturnQos> XmlBlasterAccess::erase(const EraseKey& key, const EraseQos& qos)
{
   if (log_.call()) log_.call(ME, "erase");
   if (log_.dump()) {
      log_.dump(ME, string("erase. The key:\n") + key.toXml());
      log_.dump(ME, string("erase. The Qos:\n") + qos.toXml());
   }
   return connection_->erase(key, qos);
}

string
XmlBlasterAccess::update(const string &sessionId, UpdateKey &updateKey, const unsigned char *content, long contentSize, UpdateQos &updateQos)
{
   if (log_.call()) log_.call(ME, "::update");
   if (log_.trace()) log_.trace(ME, string("update. The sessionId is '") + sessionId + "'");
   if (log_.dump()) {
      log_.dump(ME, string("update. The key:\n") + updateKey.toXml());
      log_.dump(ME, string("update. The Qos:\n") + updateQos.toXml());
   }
   if (updateClient_)
      return updateClient_->update(sessionId, updateKey, content, contentSize, updateQos);
   return "<qos><state id='OK'/></qos>";
}

std::string XmlBlasterAccess::usage()
{
   string text = string("\n");
   text += string("Choose a connection protocol:\n");
   text += string("   -protocol           Specify a protocol to talk with xmlBlaster, choose 'SOCKET' or 'IOR' depending on your compilation.\n");
   text += string("                       Current setting is '") + Global::getInstance().getProperty().getStringProperty("protocol", Global::getDefaultProtocol());
   text += string("\n\n");
   text += string("Security features:\n");
   text += string("   -Security.Client.DefaultPlugin \"gui,1.0\"\n");
   text += string("                       Force the given authentication schema, here the GUI is enforced\n");
   text += string("                       Clients can overwrite this with ConnectQos.java\n");

   return text; // std::cout << text << std::endl;
}

void XmlBlasterAccess::initFailsafe(I_ConnectionProblems* connectionProblems)
{
   if (connection_) connection_->initFailsafe(connectionProblems);
   else connectionProblems_ = connectionProblems;   
}

string XmlBlasterAccess::ping()
{
   return connection_->ping("<qos/>");
}

long XmlBlasterAccess::flushQueue()
{
   if (!connection_) {
      throw XmlBlasterException(INTERNAL_NULLPOINTER, ME + "::flushQueue", "no connection exists when trying to flush the queue: try to connect to xmlBlaster first");
   }
   return connection_->flushQueue();
}


bool XmlBlasterAccess::isConnected() const
{
   if (!connection_) return false;
   return connection_->isConnected();
}
 

}}} // namespaces


#ifdef _XMLBLASTER_CLASSTEST

#include <util/Timestamp.h>
#include <util/thread/ThreadImpl.h>

using namespace std;
using namespace org::xmlBlaster::client;
using namespace org::xmlBlaster::util::thread;

int main(int args, char* argv[])
{
    // Init the XML platform
    try
    {
       Global& glob = Global::getInstance();
       glob.initialize(args, argv);
       Log& log = glob.getLog("org.xmlBlaster.client");

       XmlBlasterAccess xmlBlasterAccess(glob);
       ConnectQos connectQos(glob);

       log.info("main", string("the connect qos is: ") + connectQos.toXml());

       ConnectReturnQos retQos = xmlBlasterAccess.connect(connectQos, NULL);
       log.info("", "Successfully connect to xmlBlaster");

       if (log.trace()) log.trace("main", "Subscribing using XPath syntax ...");
       SubscribeKey subKey(glob,"//test","XPATH");
       log.info("main", string("subscribe key: ") + subKey.toXml());
       SubscribeQos subQos(glob);
       log.info("main", string("subscribe qos: ")  + subQos.toXml());
       try {
          SubscribeReturnQos subReturnQos = xmlBlasterAccess.subscribe(subKey, subQos);
          log.info("main", string("Success: Subscribe return qos=") +
                   subReturnQos.toXml() + " done");
       }
       catch (XmlBlasterException &ex) {
          log.error("main", ex.toXml());
       }

       PublishKey pubKey(glob);
       pubKey.setOid("HelloWorld");
       pubKey.setClientTags("<test></test>");
       PublishQos pubQos(glob);
       MessageUnit msgUnit(pubKey, string("Hi"), pubQos);

       PublishReturnQos pubRetQos = xmlBlasterAccess.publish(msgUnit);
       log.info("main", string("successfully published, publish return qos: ") + pubRetQos.toXml());

       log.info("", "Successfully published a message to xmlBlaster");
       log.info("", "Sleeping");
       Timestamp delay = 10000000000ll; // 10 seconds
       Thread::sleep(delay);
   }
   catch (XmlBlasterException &ex) {
      std::cout << ex.toXml() << std::endl;
   }
   return 0;
}

#endif
