/*------------------------------------------------------------------------------
Name:      StatusQosData.h
Project:   xmlBlaster.org
Copyright: xmlBlaster.org, see xmlBlaster-LICENSE file
------------------------------------------------------------------------------*/

/**
 * Data container handling of status returned by subscribe(), unSubscribe(), erase() and ping(). 
 * <p>
 * This data holder is accessible through decorators, each of them allowing a specialized view on the data:
 * </p>
 * <ul>
 * <li>SubscribeReturnQos Returned QoS of a subscribe() invocation (Client side)</i>
 * <li>UnSubscribeReturnQos Returned QoS of a unSubscribe() invocation (Client side)</i>
 * <li>EraseReturnQos Returned QoS of an erase() invocation (Client side)</i>
 * </ul>
 * <p>
 * For the xml representation see StatusQosSaxFactory.
 * </p>
 * @see org.xmlBlaster.util.qos.StatusQosSaxFactory
 * @see org.xmlBlaster.test.classtest.qos.StatusQosFactoryTest
 * @author xmlBlaster@marcelruff.info
 * @author laghi@swissinfo.org
 */

#ifndef _UTIL_QOS_STATUSQOSDATA_H
#define _UTIL_QOS_STATUSQOSDATA_H

#include <util/xmlBlasterDef.h>
#include <util/Constants.h>
#include <string>


namespace org { namespace xmlBlaster { namespace util { namespace qos {

class Dll_Export StatusQosData
{
private:
   std::string     ME;
   org::xmlBlaster::util::Global&    global_;

   /** the state of the message, defaults to "OK" if no state is returned */
   std::string state_; // = Constants::STATE_OK;
   std::string stateInfo_;

   /** The subscription ID of a subscribe() invocation */
   std::string subscriptionId_;

   /** The key oid of a publish(), helpful if the oid was generated by xmlBlaster */
   std::string keyOid_;

   bool persistent_; // = DEFAULT_persistent;

   /** 
    * The receive timestamp (UTC time),
    * when message arrived in requestBroker.publish() method.<br />
    * In nanoseconds elapsed since midnight, January 1, 1970 UTC
    */
   Timestamp rcvTimestamp_;

   void copy(const StatusQosData& data);

public:

   /**
    * Constructs the specialized quality of service object for status informations,
    * e.g. for a return of a subscribe() call
    * <p>
    * The state defaults to Constants::STATE_OK
    * </p>
    * @param The factory which knows how to serialize and parse me
    */
   StatusQosData(org::xmlBlaster::util::Global& global);

   StatusQosData(const StatusQosData& data);

   StatusQosData operator =(const StatusQosData& data);

   /**
    * @param state The state of an update message. See Constants::java
    */
   void setState(const std::string& state);

   /**
    * Access state of message on update().
    * @return "OK", "ERROR" etc. See Constants::java
    */
   std::string getState() const;

   /**
    * True if the message is OK on update(). 
    */
   bool isOk() const;

   /**
    * True if the message was erased by timer or by a
    * client invoking erase(). 
    */
   bool isErased() const;

   /**
    * True if a timeout on this message occurred. 
    * <p />
    * Timeouts are spanned by the publisher and thrown by xmlBlaster
    * on timeout to indicate for example
    * STALE messages or any other user problem domain specific event.
    */
   bool isTimeout() const;

   /**
    * True on cluster forward problems
    */
   bool isForwardError() const;

   /**
    * @param state The human readable state text of an update message
    */
   void setStateInfo(const std::string& stateInfo);

   /**
    * Access state of message on update().
    * @return The human readable info text
    */
   std::string getStateInfo() const;

   /**
    * If Pub/Sub style update: contains the subscribe ID which caused this update
    * @param subscriptionId null if PtP message
    */
   void setSubscriptionId(const std::string& subscriptionId);

   /**
    * If Pub/Sub style update: contains the subscribe ID which caused this update
    * @return subscribeId or null if PtP message
    */
   std::string getSubscriptionId() const;

   /**
    * Access key oid. 
    * @return The unique identifier of a message
    */
   std::string getKeyOid() const;

   /**
    * Access unique oid of a message topic. 
    */
   void setKeyOid(const std::string& oid);

   /**
    * The size in bytes of the data in XML form. 
    */
   int size() const;

   /**
    * Dump state of this object into a XML ASCII std::string.
    * <br>
    * @param extraOffset indenting of tags for nice output
    * @return internal state of the status as a XML ASCII std::string
    */
   std::string toXml(const std::string& extraOffset="") const;

   /**
    * The approximate receive timestamp (UTC time),
    * when message arrived in requestBroker.publish() method.<br />
    * In milliseconds elapsed since midnight, January 1, 1970 UTC
    */
   void setRcvTimestamp(Timestamp rcvTimestamp);

   /** 
    * The approximate receive timestamp (UTC time),
    * when message arrived in requestBroker.publish() method.<br />
    * In milliseconds elapsed since midnight, January 1, 1970 UTC
    */
   Timestamp getRcvTimestamp() const;

   /**
    * Set timestamp to current time.
    */
   void touchRcvTimestamp();

   /**
    * @param persistent mark a message as persistent
    */
   void setPersistent(bool persistent);

   /**
    * @return true/false
    */
   bool isPersistent() const;
};

}}}} // namespace

#endif
